"""Flawless Nuke menu (robust minimal version).

Compatibility:
    Works on Python 3.9 and 3.10+ (avoids the `X | Y` union syntax introduced in 3.10).
    If you later drop 3.9 support you can simplify type hints to use `Path | None`.

Usage:
    Importing this module auto-builds the menu.
    Call build_menu() again to refresh after adding new files.
"""

from __future__ import annotations  # safe for 3.9+, delays evaluation of type hints

from pathlib import Path
import sys
from typing import Optional, Union, List, Set, Any
import nuke  # type: ignore


def build_menu(content_root: Optional[Union[str, Path]] = None, menu_name: str = "Flawless") -> None:
    """Create (or rebuild) the Flawless menu.

    Parameters:
        content_root: Optional path (str or Path). If None uses this file's 'flawless' folder.
        menu_name: Name shown in Nuke's toolbar.
    """
    here = Path(__file__).resolve().parent
    root = Path(content_root) if content_root else here / "flawless_nuke_tools" / "flawless"

    # Candidate icon locations.
    icon_candidates = [
        root / "icons" / "flawless-gm.png",
        here / "flawless_nuke_tools" / "icons" / "flawless-gm.png",
    ]
    icon_path = next((str(p) for p in icon_candidates if p.exists()), None)

    # Track plugin paths we add so we don't duplicate.
    plugin_paths: Set[str] = set()

    def visible(folder: Path) -> List[Path]:
        try:
            items = [p for p in folder.iterdir() if not (p.name.startswith('.') or p.name == '__pycache__')]
        except Exception:
            items = []
        return sorted(items, key=lambda p: p.name.casefold())

    def ensure_plugin(folder: Path) -> None:
        fp = str(folder)
        if fp not in plugin_paths and fp not in nuke.pluginPath():  # type: ignore
            nuke.pluginAddPath(fp)  # type: ignore
            plugin_paths.add(fp)

    def make_gizmo(path: Path) -> None:
        ensure_plugin(path.parent)
        name = path.stem
        try:
            nuke.createNode(name)  # type: ignore
        except Exception:
            try:
                nuke.load(name)  # type: ignore
                nuke.createNode(name)  # type: ignore
            except Exception:
                nuke.message(f"Failed gizmo: {name}\nFolder: {path.parent}")  # type: ignore

    def paste_nk(path: Path) -> None:
        if path.exists():
            nuke.nodePaste(str(path))  # type: ignore
        else:
            nuke.message(f"Missing file:\n{path}")  # type: ignore

    def load_toolset(path: Path) -> None:
        if path.exists():
            try:
                nuke.loadToolset(str(path))  # type: ignore
            except Exception:
                nuke.nodePaste(str(path))  # type: ignore
        else:
            nuke.message(f"Missing file:\n{path}")  # type: ignore

    def reveal(path: Path) -> None:
        import subprocess
        if sys.platform.startswith("win"):
            subprocess.Popen(["explorer", "/select,", str(path)])
        elif sys.platform == "darwin":
            subprocess.Popen(["open", "-R", str(path)])
        else:
            subprocess.Popen(["xdg-open", str(path.parent)])

    # Acquire toolbar (Nodes) or fallback to main Nuke menu.
    toolbar = nuke.toolbar("Nodes") or nuke.menu("Nuke")  # type: ignore
    top_menu = toolbar.addMenu(menu_name, icon_path) if icon_path else toolbar.addMenu(menu_name)  # type: ignore

    if not root.is_dir():
        top_menu.addCommand("Missing folder", lambda: nuke.message(f"Expected at:\n{root}"))  # type: ignore
        return

    def add(menu: Any, path: Path, treat_toolset: bool = False) -> None:
        label = path.stem
        ext = path.suffix.lower()
        is_nk = ext in {".nk", ".nuke"}
        is_giz = ext in {".gizmo", ".giz"}
        if treat_toolset and is_nk:
            menu.addCommand(label, lambda p=path: load_toolset(p))  # type: ignore
        elif is_giz:
            menu.addCommand(label, lambda p=path: make_gizmo(p))  # type: ignore
        elif is_nk:
            menu.addCommand(label, lambda p=path: paste_nk(p))  # type: ignore
        else:
            menu.addCommand(label, lambda p=path: reveal(p))  # type: ignore

    def walk(menu: Any, folder: Path, treat_toolset: bool = False) -> None:
        for entry in visible(folder):
            if entry.is_dir():
                sub = menu.addMenu(entry.name)  # type: ignore
                walk(sub, entry, treat_toolset=treat_toolset)
            else:
                add(menu, entry, treat_toolset=treat_toolset)

    skip = {"icons", "scripts"}
    for entry in visible(root):
        if entry.is_dir() and entry.name.lower() in skip:
            continue
        if entry.is_dir():
            sub_menu = top_menu.addMenu(entry.name)  # type: ignore
            walk(sub_menu, entry, treat_toolset=(entry.name.lower() == "toolsets"))
        else:
            add(top_menu, entry)  # type: ignore


# Build immediately on import.
build_menu()
